# (c) Copyright 2010-2013. CodeWeavers, Inc.

import os
import traceback
import getpass

from gi.repository import GLib
from gi.repository import Gtk
from gi.repository import GObject

import cxdiag
import cxproduct
import cxfixes
import cxwhich

import distversion
import pyop
import bottlemanagement
import cxguitools

from cxutils import cxgettext as _


package_types = ['deb', 'rpm']

class PackageBottleController:

    def __init__(self, bottleobj, transient_for):
        self.bottle = bottleobj
        self.transient_for = transient_for

        self.xml = Gtk.Builder()
        self.xml.set_translation_domain("crossover")
        self.xml.add_from_file(cxguitools.get_ui_path("packagedlg"))
        self.xml.connect_signals(self)
        self.progbar = self.xml.get_object("progbar")

        # fill in dialog defaults
        self.xml.get_object("release_entry").set_text("1")
        self.xml.get_object("packager_entry").set_text(_("Bottle Packager <%s>") % getpass.getuser())
        self.xml.get_object("product_id_entry").set_text(distversion.BUILTIN_PRODUCT_ID)

        liststore = Gtk.ListStore(GObject.TYPE_STRING)

        combo_widget = self.xml.get_object("package_type_combo")
        combo_widget.set_model(liststore)

        for t in package_types:
            liststore.set_value(liststore.append(), 0, t)

        # Try to select a reasonable default for the package type
        if cxwhich.which(os.environ["PATH"], "rpm"):
            combo_widget.set_active(1)
        else:
            combo_widget.set_active(0)

        cell = Gtk.CellRendererText()
        combo_widget.pack_start(cell, True)
        combo_widget.add_attribute(cell, "text", 0)

        self.xml.get_object("package_bottle_dialog").set_title(_("Create Package From %s") % self.bottle.name)
        self.xml.get_object("package_bottle_dialog").props.transient_for = transient_for
        self.xml.get_object("package_bottle_dialog").show()

    def cancel_clicked(self, caller):
        self.xml.get_object("package_bottle_dialog").destroy()
        self.quit_requested(caller)

    def progbar_pulse(self):
        self.progbar.pulse()
        return True

    def create_clicked(self, _widget):
        self.animateEvent = GLib.timeout_add(100, self.progbar_pulse)

        release = self.xml.get_object("release_entry").get_text()
        packager = self.xml.get_object("packager_entry").get_text()
        productID = self.xml.get_object("product_id_entry").get_text()
        pkg_type_id = self.xml.get_object("package_type_combo").get_active()
        pkg_type = package_types[pkg_type_id]
        outdir = self.xml.get_object("location_chooser").get_filename()

        self.packageOp = PackageBottleOperation(self, pkg_type, self.bottle.name, packager, release, productID, outdir)

        self.xml.get_object("package_details").set_sensitive(False)
        self.xml.get_object("package_type_combo").set_sensitive(False)
        self.xml.get_object("dialog-action_area").set_sensitive(False)
        self.progbar.show()

        pyop.sharedOperationQueue.enqueue(self.packageOp)


    def quit_requested(self, _caller):
        # pylint: disable=R0201
        cxguitools.toplevel_quit()

    def package_command_finished(self, _caller, retcode, err):
        if retcode:
            self.xml.get_object("package_bottle_dialog").destroy()
            self.quit_requested(None)
        else:
            self.xml.get_object("package_details").set_sensitive(True)
            self.xml.get_object("package_type_combo").set_sensitive(True)
            self.xml.get_object("dialog-action_area").set_sensitive(True)
            self.progbar.hide()

            title = _("Packaging failed")
            cxguitools.CXMessageDlg(primary=title, secondary=err, message_type=Gtk.MessageType.ERROR, parent=self.xml.get_object("package_bottle_dialog"))


class PackageBottleOperation(pyop.PythonOperation):

    def __init__(self, inPackageController, inType, inBottleName, inPackager, inRelease, inProductID, inOutDir):
        pyop.PythonOperation.__init__(self)

        self.bottlename = inBottleName
        self.packageController = inPackageController
        self.type = inType
        self.packager = inPackager
        self.release = inRelease
        self.productID = inProductID
        self.outDir = inOutDir


    def __unicode__(self):
        return "PackageBottleOperation for " + self.bottlename


    def main(self):
        try:
            if cxproduct.get_config_boolean('OfficeSetup', 'ApplyCxfixes', True):
                flags = cxdiag.CHECK_NOBIT
                if self.type == 'deb':
                    flags |= cxdiag.CHECK_PKGDEB
                elif self.type == 'rpm':
                    flags |= cxdiag.CHECK_PKGRPM
                if flags != cxdiag.CHECK_NOBIT:
                    diag = cxdiag.get(None, flags)
                    cxfixes.clear_errors()
                    for errid, title in diag.warnings.items():
                        cxfixes.add_error(errid, title)
                    if cxfixes.fix_errors() is not None:
                        cxdiag.clear()
                    # continue even if there are still errors

            self.retcode, self.err = bottlemanagement.package_bottle(self.bottlename, self.type, self.release, self.productID, self.packager, self.outDir)
        except:
            traceback.print_exc()


    def finish(self):
        self.packageController.package_command_finished(self, self.retcode, self.err)
        pyop.PythonOperation.finish(self)
