#############################################################################
##
#W  ideals-def.gd           Manuel Delgado <mdelgado@fc.up.pt>
#W                          Pedro A. Garcia-Sanchez <pedro@ugr.es>
#W                          Jose Morais <josejoao@fc.up.pt>
##
##
#Y  Copyright 2005 by Manuel Delgado,
#Y  Pedro Garcia-Sanchez and Jose Joao Morais
#Y  We adopt the copyright regulations of GAP as detailed in the
#Y  copyright notice in the GAP manual.
##
#############################################################################

#############################################################################
##
#R  IsIdealOfNumericalSemigroupRep
##
##  The representation of an ideal of a numerical semigroup.
##
#############################################################################
DeclareRepresentation( "IsIdealOfNumericalSemigroupRep", IsAttributeStoringRep,
        ["UnderlyingNSIdeal", "Generators"] );


#############################################################################
##
#C  IsIdealOfNumericalSemigroup
##
##  The category of ideals of numerical semigroups.
##
#############################################################################
DeclareCategory( "IsIdealOfNumericalSemigroup", IsAdditiveMagma and IsIdealOfNumericalSemigroupRep);


# Elements of ideals of numerical semigroups are integers, so ideals of
# numerical semigroups are collections of integers.
BindGlobal( "IdealsOfNumericalSemigroupsType",
        NewType( CollectionsFamily(CyclotomicsFamily),
                 IsIdealOfNumericalSemigroup));


#############################################################################
##
#F IdealOfNumericalSemigroup(l,S)
##
## l is a list of integers and S a numerical semigroup
##
## returns the ideal of S generated by l.
##
#############################################################################
DeclareGlobalFunction("IdealOfNumericalSemigroup");
#A
DeclareAttribute( "UnderlyingNSIdeal", IsNumericalSemigroup);


#############################################################################
##
#A  Generators(I)
#A  GeneratorsOfIdealOfNumericalSemigroup(I)
##
##  Returns a set of generators of the ideal I.
##  If a minimal generating system has already been computed, this
##  is the set returned.
############################################################################
##DeclareGlobalFunction("GeneratorsOfIdealOfNumericalSemigroup");
#A
DeclareAttribute( "Generators", IsIdealOfNumericalSemigroup);
DeclareSynonymAttr( "GeneratorsOfIdealOfNumericalSemigroup", Generators);


#############################################################################
##
#F  GeneratorsOfIdealOfNumericalSemigroupNC(I)
## just for compatibility of code...
##  Returns a set of generators of the ideal I.
############################################################################
DeclareSynonym("GeneratorsOfIdealOfNumericalSemigroupNC",GeneratorsOfIdealOfNumericalSemigroup);



#############################################################################
##
#F AmbientNumericalSemigroupOfIdeal(I)
##
##  Returns the ambient semigroup of the ideal I.
############################################################################
DeclareGlobalFunction("AmbientNumericalSemigroupOfIdeal");

#############################################################################
##
#P  IsIntegralIdealOfNumericalSemigroup(i)
##
##  Detects if the ideal i is contained in its ambient semigroup
##
#############################################################################
DeclareProperty("IsIntegral", IsIdealOfNumericalSemigroup);
DeclareSynonym("IsIntegralIdealOfNumericalSemigroup", IsIntegral);

#############################################################################
##
#O  IsComplementOfIntegralIdeal(X,S)
#O  IsComplementOfIntegralIdeal(S,X)
##
##  Determines if the subset X of S is the complement of an integral ideal
##  of S.
############################################################################
DeclareOperation("IsComplementOfIntegralIdeal",[IsList,IsNumericalSemigroup]);
DeclareOperation("IsComplementOfIntegralIdeal",[IsNumericalSemigroup,IsList]);

#############################################################################
##
#O  IdealByDivisorClosedSet(X,S)
#O  IdealByDivisorClosedSet(S,X)
##
##  If X is a divisor closed subset of S (for all x in X and y in S with x-y in S, 
##  the integer y is in X), then it returns the ideal S\X.
############################################################################
DeclareOperation("IdealByDivisorClosedSet",[IsList,IsNumericalSemigroup]);
DeclareOperation("IdealByDivisorClosedSet",[IsNumericalSemigroup,IsList]);



#############################################################################
##
#F SumIdealsOfNumericalSemigroup(I,J)
##
## returns the sum of the ideals I and J (in the same ambient semigroup)
#############################################################################
DeclareGlobalFunction("SumIdealsOfNumericalSemigroup");



#############################################################################
##
#F SubtractIdealsOfNumericalSemigroup(I,J)
##
## returns the ideal I - J
#############################################################################
DeclareGlobalFunction("SubtractIdealsOfNumericalSemigroup");


#############################################################################
##
#F  BelongsToIdealOfNumericalSemigroup(n,I)
##
##  Tests if the integer n belongs to the ideal I.
##
#############################################################################
DeclareGlobalFunction("BelongsToIdealOfNumericalSemigroup");



#############################################################################
##
#F DifferenceOfdealsOfNumericalSemigroup(I,J)
##
## returns the set difference I\J
#############################################################################
DeclareGlobalFunction("DifferenceOfIdealsOfNumericalSemigroup");



#############################################################################
##
#F MultipleOfIdealOfNumericalSemigroup(n,I)
##
## n is a non negative integer and I is an ideal
## returns the multiple nI (I+...+I n times) of I
#############################################################################
DeclareGlobalFunction("MultipleOfIdealOfNumericalSemigroup");



#############################################################################
##
#F HilbertFunctionOfIdealOfNumericalSemigroup(n,I)
##
## returns the value of the Hilbert function associated to I in n,
## that is, nI\(n+1)I. I must be an ideal included in its ambient semigroup.
#############################################################################
DeclareGlobalFunction("HilbertFunctionOfIdealOfNumericalSemigroup");
DeclareAttribute("HilbertFunction",IsIdealOfNumericalSemigroup);


#############################################################################
##
#F BlowUpIdealOfNumericalSemigroup(I)
##
##  Computes the Blow Up (of the maximal ideal) of
##  the numerical semigroup <s>.
#############################################################################
DeclareGlobalFunction("BlowUpIdealOfNumericalSemigroup");
DeclareOperation("BlowUp",[IsIdealOfNumericalSemigroup]);

#############################################################################
##
#A MinimalGenerators(I)
#A MinimalGeneratingSystem(I)
#A MinimalGeneratingSystemOfIdealOfNumericalSemigroup(I)
##
## The argument I is an ideal of a numerical semigroup
## returns the minimal generating system of I.
##
#############################################################################
DeclareAttribute( "MinimalGenerators", IsIdealOfNumericalSemigroup);
#DeclareSynonymAttr("MinimalGeneratingSystem", MinimalGenerators);
DeclareSynonymAttr("MinimalGeneratingSystemOfIdealOfNumericalSemigroup", MinimalGenerators);


#############################################################################
##
#A SmallElements
#A SmallElementsOfIdealOfNumericalSemigroup
##
##  Returns the list of elements in the ideal I up to the last gap + 1.
##
#############################################################################
DeclareAttribute("SmallElements", IsIdealOfNumericalSemigroup);
DeclareSynonymAttr( "SmallElementsOfIdealOfNumericalSemigroup", SmallElements);

#############################################################################
##
#A  ConductorOfIdealOfNumericalSemigroup(I)
##
##  Returns the conductor of I, the largest element in SmallElements(I)
##
#############################################################################
DeclareAttribute("Conductor", IsIdealOfNumericalSemigroup);
DeclareSynonymAttr("ConductorOfIdealOfNumericalSemigroup", Conductor);

#############################################################################
##
#A  FrobeniusNumberOfIdealOfNumericalSemigroup(I)
##
##  Returns the Frobenius number of I, the largest integer not in I
##
#############################################################################
DeclareAttribute("FrobeniusNumber", IsIdealOfNumericalSemigroup);
DeclareSynonymAttr("FrobeniusNumberOfIdealOfNumericalSemigroup", FrobeniusNumber);

#############################################################################
##
#A  PseudoFrobenius(I)
##
##  Returns the pseudo-Frobenius numbers of the ideal I, see [DS21]
##
#############################################################################
DeclareAttribute("PseudoFrobenius",IsIdealOfNumericalSemigroup);
DeclareSynonymAttr("PseudoFrobeniusOfIdealOfNumericalSemigroup",PseudoFrobenius);

#############################################################################
##
#O  Type(I)
##
##  Returns the type of the ideal I, see [DS21]
##
#############################################################################
DeclareOperation("Type",[IsIdealOfNumericalSemigroup]);


#############################################################################
##
#F ElementNumber_IdealOfNumericalSemigroup(S,n)
# Given an ideal I of a numerical semigroup and an integer n, returns 
# the nth element of I
#############################################################################
DeclareGlobalFunction("ElementNumber_IdealOfNumericalSemigroup");

#############################################################################
##
#F NumberElement_IdealOfNumericalSemigroup(S,n)
# Given an ideal of a numerical semigroup I and an integer n, returns the 
# position of n in I
#############################################################################
DeclareGlobalFunction("NumberElement_IdealOfNumericalSemigroup");



#############################################################################
##
#F  MaximalIdealOfNumericalSemigroup(S)
##
##  Returns the maximal ideal of S.
##
#############################################################################
DeclareGlobalFunction("MaximalIdealOfNumericalSemigroup");
DeclareOperation("MaximalIdeal",[IsNumericalSemigroup]);

#############################################################################
##
#P IsMonomialNumericalSemigroup
## Tests if a numerical semigroup is a monomial semigroup ring
##
#############################################################################
DeclareProperty("IsMonomialNumericalSemigroup", IsNumericalSemigroup);

#############################################################################
##
#F  BlowUpOfNumericalSemigroup(s)
##
##  Computes the Blow Up (of the maximal ideal) of
##  the numerical semigroup <s>.
##
#############################################################################
DeclareGlobalFunction("BlowUpOfNumericalSemigroup");
DeclareSynonym("LipmanSemigroup",BlowUpOfNumericalSemigroup);
DeclareOperation("BlowUp",[IsNumericalSemigroup]);

#############################################################################
##
#F MultiplicitySequenceOfNumericalSemigroup(s)
##
##  Computes the multiplicity sequence of the numerical semigroup <s>.
##
#############################################################################
DeclareGlobalFunction("MultiplicitySequenceOfNumericalSemigroup");
DeclareOperation("MultiplicitySequence",[IsNumericalSemigroup]);

#############################################################################
##
#F  MicroInvariantsOfNumericalSemigroup(s)
##
##  Computes the microinvariants of the numerical semigroup <s>
##  using the formula given by Valentina and Ralf [BF06]. The
##  microinvariants of a numerial semigroup where introduced
##  by J. Elias in [E01].
##
#############################################################################
DeclareGlobalFunction("MicroInvariantsOfNumericalSemigroup");
DeclareOperation("MicroInvariants",[IsNumericalSemigroup]);

#############################################################################
##
#P  IsGradedAssociatedRingNumericalSemigroupCM(s)
##
##  Returns true if the associated graded ring of
##  the semigroup ring algebra k[[s]] is Cohen-Macaulay.
##  This function implements the algorithm given in [BF06].
##
#############################################################################
DeclareProperty("IsGradedAssociatedRingNumericalSemigroupCM", IsNumericalSemigroup);


#############################################################################
##
#F  CanonicalIdealOfNumericalSemigroup(S)
##
##  Computes a canonical ideal of S [B06]:
##      { x in Z | g-x not in S}
##
#############################################################################
DeclareGlobalFunction("CanonicalIdealOfNumericalSemigroup");
DeclareOperation("CanonicalIdeal",[IsNumericalSemigroup]);

#############################################################################
##
#P  IsCanonicalIdealOfNumericalSemigroup(e)
##
##  Detects if the ideal e is a translation of the canonical ideal of its
##  ambient semigroup
##
#############################################################################
DeclareProperty("IsCanonicalIdeal",IsIdealOfNumericalSemigroup);
DeclareSynonymAttr("IsCanonicalIdealOfNumericalSemigroup", IsCanonicalIdeal);

#############################################################################
##
#P  IsAlmostCanonical(e)
##
##  Detects if the ideal e is almost canonical as defined in [DS21]
##
#############################################################################
DeclareProperty("IsAlmostCanonical",IsIdealOfNumericalSemigroup);

#############################################################################
##
#F  TracecalIdealOfNumericalSemigroup(s)
##
##  Computes a trace ideal of s
##
#############################################################################
DeclareGlobalFunction("TraceIdealOfNumericalSemigroup");
DeclareOperation("TraceIdeal",[IsNumericalSemigroup]);


#############################################################################
##
#A  ReductionNumberIdealNumericalSemigroup(I)
##
##  Returns the least nonnegative integer such that
##  nI-nI=(n+1)I-(n+1)I, see [B06].
##
#############################################################################
DeclareAttribute("ReductionNumber",IsIdealOfNumericalSemigroup);
DeclareSynonymAttr("ReductionNumberIdealNumericalSemigroup",ReductionNumber);

#############################################################################
##
#F  RatliffRushClosureOfIdealOfNumericalSemigroup(I)
##
##  Returns the the union of all (n+1)I-nI with n nonnegative integers
##
#############################################################################
DeclareGlobalFunction("RatliffRushClosureOfIdealOfNumericalSemigroup");
DeclareOperation("RatliffRushClosure",[IsIdealOfNumericalSemigroup]);

#############################################################################
##
#F  RatliffRushNumberOfIdealOfNumericalSemigroup(I)
##
##  Returns the least nonnegative integer such that
##  (n+1)I-nI is the Ratliff-Rush closure of I, see [DA-G-H].
##
#############################################################################
DeclareGlobalFunction("RatliffRushNumberOfIdealOfNumericalSemigroup");
DeclareOperation("RatliffRushNumber",[IsIdealOfNumericalSemigroup]); 

#############################################################################
##
#F  AsymptoticRatliffRushNumberOfIdealOfNumericalSemigroup(I)
##
##  Returns the least nonnegative integer n such that
##  mI equals the Ratliff-Rush closure of mI for all m>=n,  see [DA-G-H].
##
#############################################################################
DeclareGlobalFunction("AsymptoticRatliffRushNumberOfIdealOfNumericalSemigroup");
DeclareOperation("AsymptoticRatliffRushNumber",[IsIdealOfNumericalSemigroup]); 


#############################################################################
##
#F  TranslationOfIdealOfNumericalSemigroup(k,I)
##
##  Given an ideal <I> of a numerical semigroup S and an integer <k>
##  returns an ideal of the numerical semigroup S generated by
##  {i1+k,...,in+k} where {i1,...,in} is the system of generators of <I>.
##
#############################################################################
DeclareGlobalFunction("TranslationOfIdealOfNumericalSemigroup");


#############################################################################
##
#F  IntersectionIdealsOfNumericalSemigroup(I,J)
##
##  Given two ideals <I> and <J> of a numerical semigroup S
##  returns the ideal of the numerical semigroup S which is the
##  intersection of the ideals <I> and <J>.
##
#############################################################################
DeclareGlobalFunction("IntersectionIdealsOfNumericalSemigroup");

########################################################################
##
#F AperyListOfIdealOfNumericalSemigroupWRTElement(I,n)
##
##  Computes the sets of elements x of I such that x-n not in I,
##  where n is supposed to be in the ambient semigroup of I.
##  The element in the i-th position of the output list (starting in 0)
##  is congruent with i modulo n
########################################################################
DeclareGlobalFunction("AperyListOfIdealOfNumericalSemigroupWRTElement");

########################################################################
##
#F AperyTableOfNumericalSemigroup(S)
##
##  Computes the Apéry table associated to S as
## explained in [CJZ],
##  that is, a list containing the Apéry list of S with respect to
## its multiplicity and the Apéry lists of kM (with M the maximal
##  ideal of S) with respect to the multiplicity of S, for k=1..r,
##  where r is the reduction number of M
##  (see ReductionNumberIdealNumericalSemigroup).
########################################################################
DeclareGlobalFunction("AperyTableOfNumericalSemigroup");
DeclareOperation("AperyTable",[IsNumericalSemigroup]);

########################################################################
##
#F StarClosureOfIdealOfNumericalSemigroup(i,is)
##  i is an ideal and is is a set of ideals (all from the same
##	numerical semigroup). The output is i^{*_is}, where
## *_is is the star operation generated by is
## The implementation uses Section 3 of
##  -D. Spirito, Star Operations on Numerical Semigroups
########################################################################

DeclareGlobalFunction("StarClosureOfIdealOfNumericalSemigroup");

########################################################################
##
#O IrreducibleZComponents(i)
##
## i is an ideal of a numerical semigroup
## The output is the list of irreducible Z-components of the ideal
## There are exactly t(i) (type of i) Z-components and i is the 
## intersection of them
## See Proposition 24 in A. Assi, M. D'Anna, P. A. García-Sánchez, 
## Numerical semigroups and applications, Second edition, 
## RSME Springer series 3, Springer, Switzerland, 2020.
########################################################################
DeclareOperation("IrreducibleZComponents",[IsIdealOfNumericalSemigroup]);


########################################################################
##
#O DecomposeIntegralIdealIntoIrreducibles(i)
##
## i is an integral (proper) ideal of a numerical semigroup S
## The output is a list of irreducible ideals of S, such that its 
## intersection is the unique irredundant decompostion of i into 
## proper irreducible ideals
## The calculation is performed using Theorem 4 in 
## A. Assi, M. D'Anna, P. A. García-Sánchez, 
## Numerical semigroups and applications, Second edition, 
## RSME Springer series 3, Springer, Switzerland, 2020.
########################################################################
DeclareOperation("DecomposeIntegralIdealIntoIrreducibles",[IsIdealOfNumericalSemigroup]);
