"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.IrcProvisioningError = exports.IrcErrCode = exports.isValidListingsParams = exports.isValidUnlinkBody = exports.isValidRequestLinkBody = exports.isValidQueryLinkBody = exports.ajv = void 0;
const ajv_1 = __importDefault(require("ajv"));
exports.ajv = new ajv_1.default({
    allErrors: true,
});
const matrixRoomIdSchema = {
    type: "string",
    pattern: "^![^!]+$",
};
const remoteRoomChannelSchema = {
    type: "string",
    // According to https://www.rfc-editor.org/rfc/rfc1459#section-1.3
    // eslint-disable-next-line no-control-regex
    pattern: "^#([^:\\x00-\\x1F\\s,]){1,199}$",
};
const remoteRoomServerSchema = {
    type: "string",
    pattern: "^[a-z\\.0-9:-]+$",
};
const opNickSchema = {
    type: "string",
};
const keySchema = {
    type: "string",
    // The regex was designed with the following considerations:
    // - It cannot start with ':' because that would indicate a trailing
    //   parameter and we treat key exclusively as a middle parameter.
    // - Commas are disallowed to prevent multiple keys, as we do not support
    //   joining multiple channels simultaneously.
    // - Space is disallowed because it signifies the end of the parameter. We
    //   use \s instead of a literal space to also exclude some Unicode
    //   whitespace characters out of precaution.
    //   (see https://developer.mozilla.org/en-US/docs/Web/JavaScript/Guide/Regular_expressions/Cheatsheet)
    // - Control characters (ASCII 00-1F) are excluded to prevent issues with
    //   printing and parsing reliability. This is more stringent than
    //   recommended by RFC2812.
    pattern: "^(?!:)[^\\x00-\\x1F\\s,]*$",
    nullable: true,
};
const queryLinkBodySchema = {
    type: "object",
    properties: {
        remote_room_channel: remoteRoomChannelSchema,
        remote_room_server: remoteRoomServerSchema,
        key: keySchema,
    },
    required: [
        "remote_room_channel",
        "remote_room_server",
    ],
};
exports.isValidQueryLinkBody = exports.ajv.compile(queryLinkBodySchema);
const requestLinkBodySchema = {
    type: "object",
    properties: {
        remote_room_channel: remoteRoomChannelSchema,
        remote_room_server: remoteRoomServerSchema,
        matrix_room_id: matrixRoomIdSchema,
        op_nick: opNickSchema,
        key: keySchema,
    },
    required: [
        "remote_room_channel",
        "remote_room_server",
        "matrix_room_id",
        "op_nick",
    ],
};
exports.isValidRequestLinkBody = exports.ajv.compile(requestLinkBodySchema);
const unlinkBodySchema = {
    type: "object",
    properties: {
        remote_room_channel: remoteRoomChannelSchema,
        remote_room_server: remoteRoomServerSchema,
        matrix_room_id: matrixRoomIdSchema,
    },
    required: [
        "remote_room_channel",
        "remote_room_server",
        "matrix_room_id",
    ],
};
exports.isValidUnlinkBody = exports.ajv.compile(unlinkBodySchema);
const listingsParamsSchema = {
    type: "object",
    properties: {
        roomId: matrixRoomIdSchema,
    },
    required: [
        "roomId",
    ],
};
exports.isValidListingsParams = exports.ajv.compile(listingsParamsSchema);
var IrcErrCode;
(function (IrcErrCode) {
    IrcErrCode["UnknownNetwork"] = "IRC_UNKNOWN_NETWORK";
    IrcErrCode["UnknownChannel"] = "IRC_UNKNOWN_CHANNEL";
    IrcErrCode["UnknownRoom"] = "IRC_UNKNOWN_ROOM";
    IrcErrCode["DoubleBridge"] = "IRC_DOUBLE_BRIDGE";
    IrcErrCode["ExistingMapping"] = "IRC_EXISTING_MAPPING";
    IrcErrCode["ExistingRequest"] = "IRC_EXISTING_REQUEST";
    IrcErrCode["NotEnoughPower"] = "IRC_NOT_ENOUGH_POWER";
    IrcErrCode["BadOpTarget"] = "IRC_BAD_OPERATOR_TARGET";
    IrcErrCode["BridgeAtLimit"] = "IRC_BRIDGE_AT_LIMIT";
})(IrcErrCode || (exports.IrcErrCode = IrcErrCode = {}));
const ErrCodeToStatusCode = {
    IRC_UNKNOWN_NETWORK: 404,
    IRC_UNKNOWN_CHANNEL: 404,
    IRC_UNKNOWN_ROOM: 404,
    IRC_EXISTING_MAPPING: 409,
    IRC_EXISTING_REQUEST: 409,
    IRC_DOUBLE_BRIDGE: 409,
    IRC_NOT_ENOUGH_POWER: 403,
    IRC_BAD_OPERATOR_TARGET: 400,
    IRC_BRIDGE_AT_LIMIT: 500
};
class IrcProvisioningError extends Error {
    error;
    errcode;
    statusCode;
    additionalContent;
    constructor(error, errcode, statusCode = -1, additionalContent = {}) {
        super(`API error ${errcode}: ${error}`);
        this.error = error;
        this.errcode = errcode;
        this.statusCode = statusCode;
        this.additionalContent = additionalContent;
        if (statusCode === -1) {
            this.statusCode = ErrCodeToStatusCode[errcode];
        }
    }
    get jsonBody() {
        return {
            errcode: this.errcode,
            error: this.error,
            ...this.additionalContent,
        };
    }
    apply(response) {
        response.status(this.statusCode).send(this.jsonBody);
    }
}
exports.IrcProvisioningError = IrcProvisioningError;
//# sourceMappingURL=Schema.js.map