/*
    Copyright (c) 2005-2021 Intel Corporation

    Licensed under the Apache License, Version 2.0 (the "License");
    you may not use this file except in compliance with the License.
    You may obtain a copy of the License at

        http://www.apache.org/licenses/LICENSE-2.0

    Unless required by applicable law or agreed to in writing, software
    distributed under the License is distributed on an "AS IS" BASIS,
    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
    See the License for the specific language governing permissions and
    limitations under the License.
*/

#ifndef __TBB_blocked_range2d_H
#define __TBB_blocked_range2d_H

#include <cstddef>

#include "detail/_config.h"
#include "detail/_namespace_injection.h"
#include "detail/_range_common.h"

#include "blocked_range.h"

namespace tbb {
namespace detail {
namespace d1 {

//! A 2-dimensional range that models the Range concept.
/** @ingroup algorithms */
template<typename RowValue, typename ColValue = RowValue>
    __TBB_requires(blocked_range_value<RowValue> &&
                   blocked_range_value<ColValue>)
class blocked_range2d {
public:
    //! Type for size of an iteration range
    using row_range_type = blocked_range<RowValue>;
    using col_range_type = blocked_range<ColValue>;

private:
    row_range_type my_rows;
    col_range_type my_cols;

public:
    blocked_range2d( RowValue row_begin, RowValue row_end, typename row_range_type::size_type row_grainsize,
                     ColValue col_begin, ColValue col_end, typename col_range_type::size_type col_grainsize ) :
        my_rows(row_begin,row_end,row_grainsize),
        my_cols(col_begin,col_end,col_grainsize)
    {}

    blocked_range2d( RowValue row_begin, RowValue row_end,
                     ColValue col_begin, ColValue col_end ) :
        my_rows(row_begin,row_end),
        my_cols(col_begin,col_end)
    {}

    //! True if range is empty
    bool empty() const {
        // Range is empty if at least one dimension is empty.
        return my_rows.empty() || my_cols.empty();
    }

    //! True if range is divisible into two pieces.
    bool is_divisible() const {
        return my_rows.is_divisible() || my_cols.is_divisible();
    }

    blocked_range2d( blocked_range2d& r, split ) :
        my_rows(r.my_rows),
        my_cols(r.my_cols)
    {
        split split_obj;
        do_split(r, split_obj);
    }

    blocked_range2d( blocked_range2d& r, proportional_split& proportion ) :
        my_rows(r.my_rows),
        my_cols(r.my_cols)
    {
        do_split(r, proportion);
    }

    //! The rows of the iteration space
    const row_range_type& rows() const { return my_rows; }

    //! The columns of the iteration space
    const col_range_type& cols() const { return my_cols; }

private:
    template <typename Split>
    void do_split( blocked_range2d& r, Split& split_obj ) {
        if ( my_rows.size()*double(my_cols.grainsize()) < my_cols.size()*double(my_rows.grainsize()) ) {
            my_cols.my_begin = col_range_type::do_split(r.my_cols, split_obj);
        } else {
            my_rows.my_begin = row_range_type::do_split(r.my_rows, split_obj);
        }
    }
};

} // namespace d1
} // namespace detail

inline namespace v1 {
using detail::d1::blocked_range2d;
} // namespace v1
} // namespace tbb

#endif /* __TBB_blocked_range2d_H */
