!===============================================================================
! Copyright 2020-2022 Intel Corporation.
!
! This software and the related documents are Intel copyrighted  materials,  and
! your use of  them is  governed by the  express license  under which  they were
! provided to you (License).  Unless the License provides otherwise, you may not
! use, modify, copy, publish, distribute,  disclose or transmit this software or
! the related documents without Intel's prior written permission.
!
! This software and the related documents  are provided as  is,  with no express
! or implied  warranties,  other  than those  that are  expressly stated  in the
! License.
!===============================================================================

! Content:
!       Example of using sfftw_plan_dft_2d function on a (GPU) device using
!       the OpenMP target (offload) interface and out-of-place data I/O.
!
!*****************************************************************************

include "fftw/offload/fftw3_omp_offload.f90"

program sp_plan_dft_2d_outofplace

  use FFTW3_OMP_OFFLOAD
  use omp_lib, ONLY : omp_get_num_devices
  use, intrinsic :: ISO_C_BINDING

  include 'fftw/fftw3.f'

  ! Sizes of 2D transform
  integer, parameter :: N1 = 8
  integer, parameter :: N2 = 16

  ! Arbitrary harmonic used to verify FFT
  integer, parameter :: H1 = 1
  integer, parameter :: H2 = -N2/2

  ! Need single precision
  integer, parameter :: WP = selected_real_kind(6,37)

  ! Execution status
  integer :: statusf = 0, statusb = 0, status = 0

  ! Data array
  complex(WP), allocatable :: x(:,:), y(:, :)

  ! FFTW plan
  integer*8 :: fwd = 0, bwd = 0

  print *,"Example sp_plan_dft_2d_outofplace"
  print *,"Forward and backward 2D complex out-of-place FFT"
  print *,"Configuration parameters:"
  print '("  N = ["I0","I0"]")', N1, N2
  print '("  H = ["I0","I0"]")', H1, H2

  print *,"Allocate array for input data"
  allocate ( x(N1,N2), STAT = status)
  if (0 /= status) goto 999
  
  print *,"Allocate array for output data"
  allocate ( y(N1,N2), STAT = status)
  if (0 /= status) goto 999

  print *,"Initialize data for forward transform"
  call init(x, N1, N2, H1, H2)

  print *,"Create FFTW forward transform plan"
  !$omp target data map(tofrom:x,y)
#if defined(ONEMKL_USE_OPENMP_VERSION) && (ONEMKL_USE_OPENMP_VERSION >= 202011)
  !$omp dispatch
#else
  !$omp target variant dispatch use_device_ptr(x, y)
#endif
  call sfftw_plan_dft_2d(fwd, N1, N2, x, y, FFTW_FORWARD, FFTW_ESTIMATE)
#if !defined(ONEMKL_USE_OPENMP_VERSION) || (ONEMKL_USE_OPENMP_VERSION < 202011)
  !$omp end target variant dispatch
#endif
  if (0 == fwd) print *, "Call to sfftw_plan_dft_2d for forward transform has &
                          &failed"

  print *,"Create FFTW backward transform plan"
#if defined(ONEMKL_USE_OPENMP_VERSION) && (ONEMKL_USE_OPENMP_VERSION >= 202011)
  !$omp dispatch
#else
  !$omp target variant dispatch use_device_ptr(x, y)
#endif
  call sfftw_plan_dft_2d(bwd, N1, N2, y, x, FFTW_BACKWARD, FFTW_ESTIMATE)
#if !defined(ONEMKL_USE_OPENMP_VERSION) || (ONEMKL_USE_OPENMP_VERSION < 202011)
  !$omp end target variant dispatch
#endif
  if (0 == bwd) print *, "Call to sfftw_plan_dft_2d for backward transform has &
                          &failed"

  print *,"Compute forward transform"
#if defined(ONEMKL_USE_OPENMP_VERSION) && (ONEMKL_USE_OPENMP_VERSION >= 202011)
  !$omp dispatch
#else
  !$omp target variant dispatch
#endif
  call sfftw_execute(fwd)
#if !defined(ONEMKL_USE_OPENMP_VERSION) || (ONEMKL_USE_OPENMP_VERSION < 202011)
  !$omp end target variant dispatch
#endif

  ! Update the host with the results from forward FFT
  !$omp target update from(y)

  print *,"Verify the result of the forward transform"
  statusf = verify(y, N1, N2, H1, H2)

  print *,"Initialize data for backward transform"
  call init(y, N1, N2, -H1, -H2)

  ! Update the device with input for backward FFT
  !$omp target update to(y)

  print *,"Compute backward transform"
#if defined(ONEMKL_USE_OPENMP_VERSION) && (ONEMKL_USE_OPENMP_VERSION >= 202011)
  !$omp dispatch
#else
  !$omp target variant dispatch
#endif
  call sfftw_execute(bwd)
#if !defined(ONEMKL_USE_OPENMP_VERSION) || (ONEMKL_USE_OPENMP_VERSION < 202011)
  !$omp end target variant dispatch
#endif
  !$omp end target data

  print *,"Verify the result of the backward transform"
  statusb = verify(x, N1, N2, H1, H2)
  if ((0 /= statusf) .or. (0 /= statusb)) goto 999

100 continue

  print *,"Destroy FFTW plans"
  call sfftw_destroy_plan(fwd)
  call sfftw_destroy_plan(bwd)

  print *,"Deallocate arrays"
  deallocate(x)
  deallocate(y)

  if (status == 0) then
    print *, "TEST PASSED"
    call exit(0)
  else
    print *, "TEST FAILED"
    call exit(1)
  end if

999 print '("  Error, status forward = ",I0)', statusf
  print '(" Error, status backward = ",I0)', statusb
  status = 1
  goto 100

contains

  ! Compute mod(K*L,M) accurately
  pure integer*8 function moda(k,l,m)
    integer, intent(in) :: k,l,m
    integer*8 :: k8
    k8 = k
    moda = mod(k8*l,m)
  end function moda

  ! Initialize array with harmonic /H1, H2/
  subroutine init(x, N1, N2, H1, H2)
    integer N1, N2, H1, H2
    complex(WP) :: x(:,:)

    integer k1, k2
    complex(WP), parameter :: I_TWOPI = (0.0_WP,6.2831853071795864769_WP)

    forall (k1=1:N1, k2=1:N2)
      x(k1,k2) = exp(I_TWOPI*&
        ( real(moda(k1-1,H1,N1), WP)/N1 &
        + real(moda(k2-1,H2,N2), WP)/N2 ))/(N1*N2)
    end forall
  end subroutine init

  ! Verify that x(N1,N2) is unit peak at x(H1,H2)
  integer function verify(x, N1, N2, H1, H2)
    integer N1, N2, H1, H2
    complex(WP) :: x(:,:)

    integer k1, k2
    real(WP) err, errthr, maxerr
    complex(WP) :: res_exp, res_got

    ! Note, this simple error bound doesn't take into account error of
    ! input data
    errthr = 5.0 * log(real(N1*N2,WP)) / log(2.0_WP) * EPSILON(1.0_WP)
    print '("  Check if err is below errthr " G10.3)', errthr

    maxerr = 0.0_WP
    do k2 = 1, N2
      do k1 = 1, N1
        if (mod(k1-1-H1, N1)==0 .AND. mod(k2-1-H2, N2)==0) then
          res_exp = 1.0_WP
        else
          res_exp = 0.0_WP
        end if
        res_got = x(k1,k2)
        err = abs(res_got - res_exp)
        maxerr = max(err,maxerr)
        if (.not.(err < errthr)) then
          print '("  x("I0","I0"):"$)', k1, k2
          print '(" expected ("G14.7","G14.7"),"$)', res_exp
          print '(" got ("G14.7","G14.7"),"$)', res_got
          print '(" err "G10.3)', err
          print *," Verification FAILED"
          verify = 1
          return
        end if
      end do
    end do
    print '("  Verified,  maximum error was " G10.3)', maxerr
    verify = 0
  end function verify

end program sp_plan_dft_2d_outofplace
