/*******************************************************************************
* Copyright 2021-2022 Intel Corporation.
*
* This software and the related documents are Intel copyrighted  materials,  and
* your use of  them is  governed by the  express license  under which  they were
* provided to you (License).  Unless the License provides otherwise, you may not
* use, modify, copy, publish, distribute,  disclose or transmit this software or
* the related documents without Intel's prior written permission.
*
* This software and the related documents  are provided as  is,  with no express
* or implied  warranties,  other  than those  that are  expressly stated  in the
* License.
*******************************************************************************/

/*
*
*  Content:
*       This example demonstrates use of oneAPI Math Kernel Library (oneMKL)
*       DPCPP USM API oneapi::mkl::sparse::matmat to perform general
*       sparse matrix-sparse matrix multiplication on a SYCL device (CPU, GPU).
*
*           C = op(A) * op(B)
*
*       where op() is defined by one of
*
*           oneapi::mkl::transpose::{nontrans,trans}
*
*       It uses the full control API usage model where the user handles both memory
*       allocation for the final C Matrix, and allocation of any temporary
*       workspaces along the way. All matrices use USM device memory which
*       requires a copy of the data to host at the end for printing results.
*
*       The supported floating point data types for matmat matrix data are:
*           float
*           double
*
*
*******************************************************************************/

// stl includes
#include <algorithm>
#include <cstdlib>
#include <cstring>
#include <iostream>
#include <iterator>
#include <limits>
#include <list>
#include <vector>

#include "mkl.h"
#include "oneapi/mkl.hpp"
#include <CL/sycl.hpp>

// local includes
#include "../common/common_for_examples.hpp"
#include "common_for_sparse_examples.hpp"

//
// Main example for Sparse Matrix-Sparse Matrix Multiply consisting of
// initialization of A and B matrices through process of creating C matrix as
// the product
//
// C = op(A) * op(B)
//
// In this case, we multiply a square symmetric A by itself to have C be the
// square of A.
//
template <typename fpType, typename intType>
int run_sparse_matrix_sparse_matrix_multiply_example(const sycl::device &dev)
{
    // Initialize data for Sparse Matrix - Sparse Matrix Multiply
    oneapi::mkl::transpose opA = oneapi::mkl::transpose::trans;
    oneapi::mkl::transpose opB = oneapi::mkl::transpose::nontrans;

    oneapi::mkl::sparse::matrix_view_descr viewA = oneapi::mkl::sparse::matrix_view_descr::general;
    oneapi::mkl::sparse::matrix_view_descr viewB = oneapi::mkl::sparse::matrix_view_descr::general;
    oneapi::mkl::sparse::matrix_view_descr viewC = oneapi::mkl::sparse::matrix_view_descr::general;

    oneapi::mkl::index_base a_index = oneapi::mkl::index_base::zero;
    oneapi::mkl::index_base b_index = oneapi::mkl::index_base::zero;
    oneapi::mkl::index_base c_index = oneapi::mkl::index_base::one;

    // Catch asynchronous exceptions
    auto exception_handler = [](sycl::exception_list exceptions) {
        for (std::exception_ptr const &e : exceptions) {
            try {
                std::rethrow_exception(e);
            }
            catch (sycl::exception const &e) {
                std::cout << "Caught asynchronous SYCL "
                             "exception during sparse::gemv:\n"
                          << e.what() << std::endl;
            }
        }
    };

    // create execution main_queue
    sycl::queue main_queue(dev, exception_handler);
    sycl::context cxt = main_queue.get_context();

    //
    // set up dimensions of matrix products
    //
    intType size = 4;

    intType a_nrows = size * size * size;
    intType a_ncols = a_nrows;
    intType a_nnz   = 27 * a_nrows;
    intType b_nrows = size * size * size;
    intType b_ncols = b_nrows;
    intType b_nnz   = 27 * b_nrows;
    intType c_nrows = size * size * size;
    intType c_ncols = c_nrows;
    // c_nnz is unknown at this point

    //
    // array memory management tools
    //
    std::vector<intType *> int_ptr_vec;
    std::vector<fpType *> fp_ptr_vec;
    std::vector<std::int64_t *> i64_ptr_vec;
    std::vector<void *> void_ptr_vec;

    //
    // setup A data locally in CSR format
    //
    intType *a_rowptr_host = sycl::malloc_host<intType>(a_nrows + 1, main_queue);
    intType *a_colind_host = sycl::malloc_host<intType>(27 * a_nrows, main_queue);
    fpType *a_values_host  = sycl::malloc_host<fpType>(27 * a_nrows, main_queue);
    if (!a_rowptr_host || !a_colind_host || !a_values_host)
        throw std::runtime_error("Failed to allocate USM memory");

    int_ptr_vec.push_back(a_rowptr_host);
    int_ptr_vec.push_back(a_colind_host);
    fp_ptr_vec.push_back(a_values_host);

    intType a_ind = a_index == oneapi::mkl::index_base::zero ? 0 : 1;
    generate_sparse_matrix<fpType, intType>(size, a_rowptr_host, a_colind_host, a_values_host, a_ind);
    a_nnz = a_rowptr_host[a_nrows] - a_ind;

    intType *a_rowptr = sycl::malloc_device<intType>(a_nrows + 1, main_queue);
    intType *a_colind = sycl::malloc_device<intType>(a_nnz, main_queue);
    fpType *a_values  = sycl::malloc_device<fpType>(a_nnz, main_queue);

    if (!a_rowptr || !a_colind || !a_values)
        throw std::runtime_error("Failed to allocate USM memory");

    int_ptr_vec.push_back(a_rowptr);
    int_ptr_vec.push_back(a_colind);
    fp_ptr_vec.push_back(a_values);

    // copy A matrix USM data from host to device
    auto ev_cpy_ia = main_queue.copy<intType>(a_rowptr_host, a_rowptr, a_nrows + 1);
    auto ev_cpy_ja = main_queue.copy<intType>(a_colind_host, a_colind, a_nnz);
    auto ev_cpy_a  = main_queue.copy<fpType>(a_values_host, a_values, a_nnz);

    //
    // setup B data locally in CSR format
    //
    intType *b_rowptr_host = sycl::malloc_host<intType>(b_nrows + 1, main_queue);
    intType *b_colind_host = sycl::malloc_host<intType>(27 * b_nrows, main_queue);
    fpType *b_values_host  = sycl::malloc_host<fpType>(27 * b_nrows, main_queue);
    if (!b_rowptr_host || !b_colind_host || !b_values_host)
        throw std::runtime_error("Failed to allocate USM memory");

    int_ptr_vec.push_back(b_rowptr_host);
    int_ptr_vec.push_back(b_colind_host);
    fp_ptr_vec.push_back(b_values_host);

    intType b_ind = b_index == oneapi::mkl::index_base::zero ? 0 : 1;
    generate_sparse_matrix<fpType, intType>(size, b_rowptr_host, b_colind_host, b_values_host, b_ind);
    b_nnz = b_rowptr_host[b_nrows] - b_ind;

    intType *b_rowptr = sycl::malloc_device<intType>(b_nrows + 1, main_queue);
    intType *b_colind = sycl::malloc_device<intType>(b_nnz, main_queue);
    fpType *b_values  = sycl::malloc_device<fpType>(b_nnz, main_queue);

    if (!a_rowptr || !b_colind || !b_values)
        throw std::runtime_error("Failed to allocate USM memory");

    int_ptr_vec.push_back(b_rowptr);
    int_ptr_vec.push_back(b_colind);
    fp_ptr_vec.push_back(b_values);

    // copy B matrix USM data from host to device
    auto ev_cpy_ib = main_queue.copy<intType>(b_rowptr_host, b_rowptr, b_nrows + 1);
    auto ev_cpy_jb = main_queue.copy<intType>(b_colind_host, b_colind, b_nnz);
    auto ev_cpy_b  = main_queue.copy<fpType>(b_values_host, b_values, b_nnz);

    //
    // setup C data locally in CSR format
    // note: we don't know c_nnz so can only prepare ic at this point
    //
    intType c_ind = c_index == oneapi::mkl::index_base::zero ? 0 : 1;
    intType *c_rowptr = sycl::malloc_device<intType>(c_nrows + 1, main_queue);
    intType *c_colind = nullptr;
    fpType *c_values = nullptr;

    if (!c_rowptr)
        throw std::runtime_error("Failed to allocate USM memory");
    int_ptr_vec.push_back(c_rowptr);

    //
    // other workspaces and arrays for matmat process
    //
    std::int64_t *sizeTempBuffer = nullptr, *sizeTempBuffer2 = nullptr, *c_nnz = nullptr;
    void *tempBuffer = nullptr, *tempBuffer2 = nullptr;

    //
    // Execute Matrix Multiply
    //

    std::cout << "\n\t\tsparse::matmat parameters:\n";
    std::cout << "\t\t\topA = " << opA << std::endl;
    std::cout << "\t\t\topB = " << opB << std::endl;

    std::cout << "\t\t\tviewA = " << viewA << std::endl;
    std::cout << "\t\t\tviewB = " << viewB << std::endl;
    std::cout << "\t\t\tviewC = " << viewC << std::endl;

    std::cout << "\t\t\tA_nrows = A_ncols = " << a_nrows << std::endl;
    std::cout << "\t\t\tB_nrows = B_ncols = " << b_nrows << std::endl;
    std::cout << "\t\t\tC_nrows = C_ncols = " << c_nrows << std::endl;

    std::cout << "\t\t\tA_index = " << a_index << std::endl;
    std::cout << "\t\t\tB_index = " << b_index << std::endl;
    std::cout << "\t\t\tC_index = " << c_index << std::endl;

    oneapi::mkl::sparse::matrix_handle_t A = nullptr;
    oneapi::mkl::sparse::matrix_handle_t B = nullptr;
    oneapi::mkl::sparse::matrix_handle_t C = nullptr;

    oneapi::mkl::sparse::matmat_descr_t descr = nullptr;
    oneapi::mkl::sparse::matmat_request req;

    try {
        oneapi::mkl::sparse::init_matrix_handle(&A);
        oneapi::mkl::sparse::init_matrix_handle(&B);
        oneapi::mkl::sparse::init_matrix_handle(&C);

        auto ev_setA = oneapi::mkl::sparse::set_csr_data(main_queue, A, a_nrows, a_ncols, a_index,
                a_rowptr, a_colind, a_values, {ev_cpy_ia, ev_cpy_ja, ev_cpy_a});
        auto ev_setB = oneapi::mkl::sparse::set_csr_data(main_queue, B, b_nrows, b_ncols, b_index,
                b_rowptr, b_colind, b_values, {ev_cpy_ib, ev_cpy_jb, ev_cpy_b});

        //
        // only c_rowptr exists at this point in process so pass in nullptrs
        //
        auto ev_setC = oneapi::mkl::sparse::set_csr_data(main_queue, C, c_nrows, c_ncols, c_index, c_rowptr,
                                                        (intType *)nullptr, (fpType *)nullptr, {});

        //
        // initialize the matmat descriptor
        //
        oneapi::mkl::sparse::init_matmat_descr(&descr);
        oneapi::mkl::sparse::set_matmat_data(descr, viewA, opA, viewB, opB, viewC);

        //
        // Stage 1:  work estimation
        //

        // Step 1.1
        //   query for size of work_estimation temp buffer
        req = oneapi::mkl::sparse::matmat_request::get_work_estimation_buf_size;
        sizeTempBuffer = sycl::malloc_host<std::int64_t>(1, main_queue);
        if (!sizeTempBuffer)
            throw std::runtime_error("Failed to allocate USM memory");
        i64_ptr_vec.push_back(sizeTempBuffer);

        auto ev1_1 = oneapi::mkl::sparse::matmat(main_queue, A, B, C, req, descr, sizeTempBuffer,
                                                 nullptr, {ev_setA, ev_setB, ev_setC});

        // Step 1.2
        //   allocate temp buffer for work_estimation
        ev1_1.wait();
        void *tempBuffer = sycl::malloc_device(sizeTempBuffer[0] * sizeof(std::uint8_t), main_queue);
        if (!tempBuffer)
            throw std::runtime_error("Failed to allocate USM memory");
        void_ptr_vec.push_back(tempBuffer);

        // Step 1.3  do work_estimation
        req = oneapi::mkl::sparse::matmat_request::work_estimation;
        auto ev1_3 = oneapi::mkl::sparse::matmat(main_queue, A, B, C, req, descr, sizeTempBuffer,
                                                 tempBuffer, {ev1_1});

        //
        // Stage 2:  compute
        //

        // Step 2.1 query size of compute temp buffer
        req = oneapi::mkl::sparse::matmat_request::get_compute_buf_size;
        sizeTempBuffer2 = sycl::malloc_host<std::int64_t>(1, main_queue);
        if (!sizeTempBuffer2)
            throw std::runtime_error("Failed to allocate USM memory");
        i64_ptr_vec.push_back(sizeTempBuffer2);
        auto ev2_1 = oneapi::mkl::sparse::matmat(main_queue, A, B, C, req, descr, sizeTempBuffer2,
                                                 nullptr, {ev1_3});

        // Step 2.2 allocate temp buffer for compute
        ev2_1.wait();
        void *tempBuffer2 = sycl::malloc_device(sizeTempBuffer2[0] * sizeof(std::uint8_t), main_queue);
        if (!tempBuffer2)
            throw std::runtime_error("Failed to allocate USM memory");
        void_ptr_vec.push_back(tempBuffer2);

        // Step 2.3 do compute
        req = oneapi::mkl::sparse::matmat_request::compute;
        auto ev2_3 = oneapi::mkl::sparse::matmat(main_queue, A, B, C, req, descr, sizeTempBuffer2,
                                                 tempBuffer2, {ev2_1});

        //
        // Stage 3:  finalize
        //

        // Step 3.1  get nnz
        req = oneapi::mkl::sparse::matmat_request::get_nnz;
        std::int64_t *c_nnz = sycl::malloc_host<std::int64_t>(1, main_queue);
        if (!c_nnz)
            throw std::runtime_error("Failed to allocate USM memory");
        i64_ptr_vec.push_back(c_nnz);
        auto ev3_1 = oneapi::mkl::sparse::matmat(main_queue, A, B, C, req, descr, c_nnz, nullptr,
                                                 {ev2_3});

        // Step 3.2  allocate final c matrix arrays
        ev3_1.wait();
        c_colind = sycl::malloc_device<intType>(c_nnz[0], main_queue);
        c_values = sycl::malloc_device<fpType>(c_nnz[0], main_queue);
        if (!c_colind || !c_values) {
            throw std::runtime_error("Failed to allocate USM memory");
        }
        int_ptr_vec.push_back(c_colind);
        fp_ptr_vec.push_back(c_values);
        ev_setC = oneapi::mkl::sparse::set_csr_data(main_queue, C, c_nrows, c_ncols, c_index, c_rowptr, c_colind, c_values, {ev3_1});

        // Step 3.3  finalize into C matrix
        req = oneapi::mkl::sparse::matmat_request::finalize;
        auto ev3_3 = oneapi::mkl::sparse::matmat(main_queue, A, B, C, req, descr, nullptr, nullptr,
                                                 {ev_setC});

        // Sort C matrix output if desired
        auto ev_sort = oneapi::mkl::sparse::sort_matrix(main_queue, C, {ev3_3});

        // Copy first set of rows of C to host for printing
        const intType c_nrows_copy = std::min<intType>(2, c_nrows); // only copy over this many rows of C to host

        intType *c_rowptr_host = sycl::malloc_host<intType>(c_nrows_copy+1, main_queue);
        if (!c_rowptr_host)
            throw std::runtime_error("Failed to allocate USM memory");
        int_ptr_vec.push_back(c_rowptr_host);

        auto ev_cpy_ic = main_queue.copy<intType>(c_rowptr, c_rowptr_host, c_nrows_copy+1, {ev_sort}); // copy part of c_rowptr device to host

        ev_cpy_ic.wait(); // make sure copy is done before reading from it
        const intType c_host_nnz = c_rowptr_host[c_nrows_copy] - c_ind;
        intType *c_colind_host = sycl::malloc_host<intType>(c_host_nnz, main_queue);
        fpType *c_values_host  = sycl::malloc_host<fpType>(c_host_nnz, main_queue);

        if (!c_colind_host || !c_values_host)
            throw std::runtime_error("Failed to allocate USM memory");

        int_ptr_vec.push_back(c_colind_host);
        fp_ptr_vec.push_back(c_values_host);

        auto ev_cpy_jc = main_queue.copy<intType>(c_colind, c_colind_host, c_host_nnz, {ev_sort, ev_cpy_ic}); // copy part of c_colind device to host
        auto ev_cpy_c  = main_queue.copy<fpType>(c_values, c_values_host, c_host_nnz, {ev_sort, ev_cpy_ic}); // copy part of c_values device to host

        // print out a portion of C solution
        sycl::event ev_print = main_queue.submit([&](sycl::handler &cgh) {
            cgh.depends_on({ev_cpy_ic, ev_cpy_jc, ev_cpy_c});
            auto kernel = [=]() {
                std::cout << "C matrix [first " << c_nrows_copy << " rows]:" << std::endl;
                for (intType row = 0; row < c_nrows_copy; ++row) {
                    for (intType j = c_rowptr_host[row] - c_ind; j < c_rowptr_host[row + 1] - c_ind; ++j) {
                        intType col = c_colind_host[j];
                        fpType val  = c_values_host[j];
                        std::cout << "C(" << row + c_ind << ", " << col << ") = " << val
                                  << std::endl;
                    }
                }
            };
            cgh.host_task(kernel);
        });

        // clean up
        oneapi::mkl::sparse::release_matmat_descr(&descr);
        auto ev_relA = oneapi::mkl::sparse::release_matrix_handle(main_queue, &A, {ev3_3});
        auto ev_relB = oneapi::mkl::sparse::release_matrix_handle(main_queue, &B, {ev3_3});
        auto ev_relC = oneapi::mkl::sparse::release_matrix_handle(main_queue, &C, {ev_print});

        ev_relA.wait();
        ev_relB.wait();
        ev_relC.wait();
    }
    catch (sycl::exception const &e) {
        std::cout << "\t\tCaught synchronous SYCL exception:\n" << e.what() << std::endl;

        main_queue.wait();
        oneapi::mkl::sparse::release_matmat_descr(&descr);
        oneapi::mkl::sparse::release_matrix_handle(main_queue, &A).wait();
        oneapi::mkl::sparse::release_matrix_handle(main_queue, &B).wait();
        oneapi::mkl::sparse::release_matrix_handle(main_queue, &C).wait();

        cleanup_arrays<fpType, intType>(fp_ptr_vec, int_ptr_vec, i64_ptr_vec, void_ptr_vec, cxt);

        return 1;
    }
    catch (std::exception const &e) {
        std::cout << "\t\tCaught std exception:\n" << e.what() << std::endl;

        main_queue.wait();
        oneapi::mkl::sparse::release_matmat_descr(&descr);
        oneapi::mkl::sparse::release_matrix_handle(main_queue, &A).wait();
        oneapi::mkl::sparse::release_matrix_handle(main_queue, &B).wait();
        oneapi::mkl::sparse::release_matrix_handle(main_queue, &C).wait();

        cleanup_arrays<fpType, intType>(fp_ptr_vec, int_ptr_vec, i64_ptr_vec, void_ptr_vec, cxt);

        return 1;
    }

    cleanup_arrays<fpType, intType>(fp_ptr_vec, int_ptr_vec, i64_ptr_vec, void_ptr_vec, cxt);

    return 0;
}

//
// Description of example setup, apis used and supported floating point type
// precisions
//
void print_example_banner()
{

    std::cout << "" << std::endl;
    std::cout << "###############################################################"
                 "#########"
              << std::endl;
    std::cout << "# Sparse Matrix-Sparse Matrix Multiply Example: " << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "#    C = op(A) * op(B)" << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "# where A and B are sparse matrices in CSR format, and C is the\n"
                 "# sparse matrix product in CSR format"
              << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "# Using apis:" << std::endl;
    std::cout << "#   sparse::matmat" << std::endl;
    std::cout << "#   sparse::init_matmat_descr" << std::endl;
    std::cout << "#   sparse::set_matmat_data" << std::endl;
    std::cout << "#   sparse::release_matmat_descr" << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "#   sparse::init_matrix_handle" << std::endl;
    std::cout << "#   sparse::set_csr_data" << std::endl;
    std::cout << "#   sparse::release_matrix_handle" << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "# Supported floating point type precisions:" << std::endl;
    std::cout << "#   float" << std::endl;
    std::cout << "#   double" << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "###############################################################"
                 "#########"
              << std::endl;
    std::cout << std::endl;
}

//
// Main entry point for example.
//
// Dispatches to appropriate device types as set at build time with flag:
// -DSYCL_DEVICES_cpu -- only runs SYCL CPU implementation
// -DSYCL_DEVICES_gpu -- only runs SYCL GPU implementation
// -DSYCL_DEVICES_all (default) -- runs on all: cpu and gpu devices
//
//  For each device selected and each supported data type, MatrixMultiplyExample
//  runs is with all supported data types
//

int main(int argc, char **argv)
{

    print_example_banner();

    std::list<my_sycl_device_types> list_of_devices;
    set_list_of_devices(list_of_devices);

    int status = 0;
    for (auto it = list_of_devices.begin(); it != list_of_devices.end(); ++it) {

        sycl::device my_dev;
        bool my_dev_is_found = false;
        get_sycl_device(my_dev, my_dev_is_found, *it);

        if (my_dev_is_found) {
            std::cout << "Running tests on " << sycl_device_names[*it] << ".\n";

            std::cout << "\tRunning with single precision real data type:" << std::endl;
            status = run_sparse_matrix_sparse_matrix_multiply_example<float, std::int32_t>(my_dev);
            if (status != 0)
                return status;

            if (my_dev.get_info<sycl::info::device::double_fp_config>().size() != 0) {
                std::cout << "\tRunning with double precision real data type:" << std::endl;
                status = run_sparse_matrix_sparse_matrix_multiply_example<double, std::int32_t>(
                        my_dev);
                if (status != 0)
                    return status;
            }
        }
        else {
#ifdef FAIL_ON_MISSING_DEVICES
            std::cout << "No " << sycl_device_names[*it]
                      << " devices found; Fail on missing devices "
                         "is enabled.\n";
            return 1;
#else
            std::cout << "No " << sycl_device_names[*it] << " devices found; skipping "
                      << sycl_device_names[*it] << " tests.\n";
#endif
        }
    }

    mkl_free_buffers();
    return status;
}
